/* This is a mmap wrapper which tries to deal with the different page sizes
 * between Intel and Alpha.
 *
 * The basic problem is that Intel implements 4K page granularity, while
 * Alpha implements 8K page granularity.  Attempting to mmap on a 4K
 * page boundary will fail on Alpha.  Rounding the boundaries out to
 * the nearest 8K can have disastrous consequences; if there was already
 * something occupying the memory where we enlarged our mmap area to,
 * we could smash it.  Page protections are a nightmare.
 *
 * The way we deal with it, ugly is as it is, is to keep track of
 * what we've mmap'ed.  Every time we do an allocation we check this
 * list.  If rounding out causes us to smash into a prevously-
 * allocated area, we round *IN* and deal with the odd half-page
 * explicitly.
 *
 */
#include <stdio.h>
#include <unistd.h>
#include <sys/mman.h>
#include <errno.h>
#include "linux_bin.h"

#define X86_PAGE_SIZE	4096
#define X86_PAGE_ROUND_UP(n)	(((n)+(X86_PAGE_SIZE-1)) & ~(X86_PAGE_SIZE-1))
#define X86_PAGE_ROUND_DOWN(n)	((n) & ~(X86_PAGE_SIZE-1))

#define ALPHA_PAGE_SIZE   8192
#define ALPHA_PAGE_ROUND_UP(n) (((n)+(ALPHA_PAGE_SIZE-1))&~(ALPHA_PAGE_SIZE-1))
#define ALPHA_PAGE_ROUND_DOWN(n)  ((n) & ~(ALPHA_PAGE_SIZE-1))

#define PAGE_VALID	0x80	/* "valid" flag for page bits */
#define PROT_BITS	(PROT_READ|PROT_WRITE|PROT_EXEC)

extern int debug_mmap;
extern FILE* x86_logfile;

extern unsigned long stktop;

static char tmpstr[512];

char * print_prot(int p)
{
    char s[80];

    strcpy(tmpstr, "");

    if(p & PAGE_VALID) {
	strcat(tmpstr, "V");
	p &= ~PAGE_VALID;
    }
    if(p & PROT_READ) {
	strcat(tmpstr, "R");
	p &= ~PROT_READ;
    }
    if(p & PROT_WRITE) {
	strcat(tmpstr, "W");
	p &= ~PROT_WRITE;
    }
    if(p & PROT_EXEC) {
	strcat(tmpstr, "X");
	p &= ~PROT_EXEC;
    }
    if(p) {
	sprintf(s, "[undef: 0x%x] ", p);
	strcat(tmpstr, s);
    }

    return(tmpstr);
}

char * print_flags(int f)
{
    int first = 1;
    char s[80];

    strcpy(tmpstr, "");

    if(f & MAP_SHARED) {
	strcat(tmpstr, "MAP_SHARED ");
	first = 0;
	f &= ~MAP_SHARED;
    }
    if(f & MAP_PRIVATE) {
	sprintf(s, "%sMAP_PRIVATE ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_PRIVATE;
    }
    if(f & MAP_FIXED) {
	sprintf(s, "%sMAP_FIXED ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_FIXED;
    }
    if(f & MAP_ANONYMOUS) {
	sprintf(s, "%sMAP_ANONYMOUS ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_ANONYMOUS;
    }
    if(f & MAP_HASSEMAPHORE) {
	sprintf(s, "%sMAP_HASSEMAPHORE ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_HASSEMAPHORE;
    }
    if(f & MAP_INHERIT) {
	sprintf(s, "%sMAP_INHERIT ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_INHERIT;
    }
    if(f & MAP_UNALIGNED) {
	sprintf(s, "%sMAP_UNALIGNED ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_UNALIGNED;
    }
    if(f & MAP_GROWSDOWN) {
	sprintf(s, "%sMAP_GROWSDOWN ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_GROWSDOWN;
    }
    if(f & MAP_DENYWRITE) {
	sprintf(s, "%sMAP_DENYWRITE ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_DENYWRITE;
    }
    if(f & MAP_EXECUTABLE) {
	sprintf(s, "%sMAP_EXECUTABLE ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_EXECUTABLE;
    }
    if(f & MAP_LOCKED) {
	sprintf(s, "%sMAP_LOCKED ", first ? "" : "| ");
	strcat(tmpstr, s);
	first = 0;
	f &= ~MAP_LOCKED;
    }
    if(f) {
	sprintf(s, "[undef: 0x%x] ", f);
	strcat(tmpstr, s);
    }
    return(tmpstr);
}

void * sys_mmap(unsigned long start, unsigned long length, int prot,
		int flags, int fd, unsigned long offset)
{
    if(debug_mmap) {
        fprintf(x86_logfile, "MMAP: Calling system mmap(0x%lx, 0x%lx,\n", start, length);
	fprintf(x86_logfile, "      0x%x (%s),\n", prot, print_prot(prot));
	fprintf(x86_logfile, "      0x%x (%s),\n", flags, print_flags(flags));
	fprintf(x86_logfile, "      %d, 0x%x)\n", fd, offset);
	fflush(x86_logfile);
    }
    return(mmap(start, length, prot, flags, fd, offset));
}

int sys_mprotect(unsigned long addr, unsigned long len, int prot)
{
    int retval;
    int addr8k;
    if(debug_mmap) {
    	fprintf(x86_logfile, "MMAP: Calling mprotect(0x%lx, 0x%lx,\n      0x%x(%s))\n",
		addr, len, prot, print_prot(prot));
	fflush(x86_logfile);
    }

    /* If the base address is in the middle of an Alpha page, then check
     * with the other half of that page as well...
     */
    addr8k = ALPHA_PAGE_ROUND_DOWN(addr);
    if(addr8k != addr) {
	retval = mprotect(addr8k, ALPHA_PAGE_SIZE, 
				prot | (get_page_bits(addr8k) & PROT_BITS));
	if((retval == -1) && debug_mmap) {
		perror("mprotect");
	}
        if(retval == -1) {
	    return(retval);
	}
	put_page_bits(addr, prot | PAGE_VALID);

	if(len <= X86_PAGE_SIZE) {
	    /* We're done! */
	    return(retval);
	}
	else {
	    addr8k = ALPHA_PAGE_ROUND_UP(addr);
	    len -= X86_PAGE_SIZE;
	}
    }

    /* At this point, we know the base address is 8K aligned.  If the
     * puts the end of this into the first half of an alpha page, we
     * must do the page-pair bit again...
     */
    if(ALPHA_PAGE_ROUND_UP(len) != X86_PAGE_ROUND_UP(len)) {
	unsigned long	frag_start;

	/* This means we're in the first half of the page.
	 * Calculate the start of this fragment, find the
	 * protections of its page-partner, and OR them
	 * together.
	 */
	frag_start = ALPHA_PAGE_ROUND_DOWN(addr + len - 1);
	retval = mprotect(frag_start, ALPHA_PAGE_SIZE, 
		    prot|(get_page_bits(frag_start+X86_PAGE_SIZE) &PROT_BITS));
	if((retval == -1) && debug_mmap) {
		perror("mprotect");
	}
        if(retval == -1) {
	    return(retval);
	}
	put_page_bits(frag_start, prot | PAGE_VALID);

	if(frag_start <= addr8k) {
	    /* This frag was all there was... */
	    return(retval);
	}
	else {
	    len = frag_start - addr8k;
	}
    }

    /* Okay; we can now do the 8k-aligned chunk in the middle. */
    retval = mprotect(addr8k, len, prot);
    if((retval == -1) && debug_mmap) {
	    perror("mprotect");
    }
    set_page_range_bits(addr8k, addr8k+len-1, prot | PAGE_VALID);
    return(retval);
    
}


void * mmap4k(unsigned long start, unsigned long length, int prot, int flags,
		int fd, unsigned long offset)
{
    unsigned long	end = start + length - 1;
    unsigned long	end4k, end8k;
    unsigned long	start4k, start8k;
    unsigned long	mmap_result;
    char		pagebits;
    char		protbits;

    if(debug_mmap) {
	fprintf(x86_logfile, "-------------------------------------\n");
        fprintf(x86_logfile, "MMAP:(0x%lx, 0x%lx,\n", start, length);
	fprintf(x86_logfile, "      0x%x (%s),\n", prot, print_prot(prot));
	fprintf(x86_logfile, "      0x%x (%s),\n", flags, print_flags(flags));
	fprintf(x86_logfile, "      %d, 0x%x)\n", fd, offset);
	fflush(x86_logfile);
    }

    /* If it's not a fixed mapping, then we'll NEVER smash into
     * anything!  Page-align the start address and have at it!
     */
    if(!(flags & MAP_FIXED)) {
	unsigned long newstart;

	if(debug_mmap) {
	    fprintf(x86_logfile, "MMAP: Performing non-fixed mapping\n");
	    fflush(x86_logfile);
	}
	newstart = sys_mmap(ALPHA_PAGE_ROUND_UP(start), length, prot, flags,
				fd, offset);
	if((long)newstart == -1) {
	    perror("non-fixed mmap");
	    if(debug_mmap) {
		fprintf(x86_logfile, "-------------------------------------\n");
		fflush(x86_logfile);
	    }
	    return (-1);
	}
	else {
	    set_page_range_bits(newstart,ALPHA_PAGE_ROUND_UP(newstart+length)-1, 
				prot | PAGE_VALID);
	    if(debug_mmap) {
	        fprintf(x86_logfile, "MMAP: Returning 0x%lx\n\n", newstart);
		fprintf(x86_logfile, "-------------------------------------\n");
	        fflush(x86_logfile);
	    }
	    return newstart;
	}
    }

    /* It's a fixed mapping.  Do it. */
    if(start != X86_PAGE_ROUND_DOWN(start)) {
	/* Fixed mappings MUST be page-aligned! */
	if(debug_mmap) {
	    fprintf(x86_logfile, "MMAP: 0x%lx not page-aligned, returning error\n", start);
	    fprintf(x86_logfile, "-------------------------------------\n");
	    fflush(x86_logfile);
	}
	errno=EINVAL;
	return(-1);
    }

    /* Calculate the 4K and 8K boundaries of this allocation */
    start4k = start;
    start8k = ALPHA_PAGE_ROUND_DOWN(start);
    end4k = X86_PAGE_ROUND_UP(end)-1;
    end8k = ALPHA_PAGE_ROUND_UP(end)-1;

    if(debug_mmap) {
	fprintf(x86_logfile, 
		"MMAP: start4k 0x%x, start8k 0x%x, end4k 0x%x, end8k 0x%x\n",
		start4k, start8k, end4k, end8k);
	fflush(x86_logfile);
    }

    if(start8k != start4k) {
	/* We have a leading frag.  Find out if we already have a
	 * valid page at that location.  If not, create one.
	 */
	pagebits = get_page_bits(start8k);
	if(!(pagebits & PAGE_VALID)) {
	    /* No, we don't.  Create a page here */
	    if(debug_mmap) {
		fprintf(x86_logfile, "Creating space for leading frag @0x%x\n",
				start8k);
		fflush(x86_logfile);
	    }
	    mmap_result = sys_mmap(start8k, ALPHA_PAGE_SIZE, prot, 
				flags | MAP_ANONYMOUS, -1, 0);
	    if((long)mmap_result == -1) {
		perror("leading frag mmap");	
		if(debug_mmap) {
	            fprintf(x86_logfile, "-------------------------------------\n");
	            fflush(x86_logfile);
		}
		return(-1);
	    }
	    pagebits = prot | PAGE_VALID;
	    put_page_bits(start8k, pagebits);
	    put_page_bits(start8k+X86_PAGE_SIZE, pagebits);
	}

	protbits = pagebits & PROT_BITS;

	/* If this is a file mapping, then we have to read in the data. */
	if(!(flags & MAP_ANONYMOUS)) {
	    int readlen = (length<X86_PAGE_SIZE) ? length : X86_PAGE_SIZE;
	    if(!(pagebits & PROT_WRITE)) {
	        sys_mprotect(start8k, ALPHA_PAGE_SIZE, protbits | PROT_WRITE);
	    }
	    if(debug_mmap) {
		fprintf(x86_logfile, 
                   "Reading in leading frag; offset 0x%x addr 0x%x len 0x%x\n", 
		   offset, start4k, readlen);
		fflush(x86_logfile);
	    }
	    lseek(fd, offset, SEEK_SET);
	    read(fd, start4k, readlen);
	    /* Page permissions on the new page should be a combination
	     * of the old and new...
	     */
	    if((protbits | PROT_WRITE) != (protbits | prot)) {
		sys_mprotect(start8k, ALPHA_PAGE_SIZE, protbits | prot);
	    }
	}
	else {
	    if(protbits != (protbits | prot)) {
		sys_mprotect(start8k, ALPHA_PAGE_SIZE, protbits | prot);
	    }
	}

	/* The length of the allocation can now be shortened */
	if(length < X86_PAGE_SIZE) {
	    /* we're done! */
	    if(debug_mmap) {
		fprintf(x86_logfile, "-------------------------------------\n");
		fflush(x86_logfile);
	    }
	    return(start);
	}
	else {
	    length -= X86_PAGE_SIZE;
	    if(!(flags & MAP_ANONYMOUS)) {
		offset += X86_PAGE_SIZE;
	    }
	}

	/* Our 8K allocation can start at the next page now */
	start8k += ALPHA_PAGE_SIZE;
	if(debug_mmap) {
	    fprintf(x86_logfile, "After dealing w/leading frag:\n");
	    fprintf(x86_logfile, "  offset=0x%x, length=0x%x, start8k=0x%x\n",
			offset, length, start8k);
	}
    }

    /* At this point, we know we're set up to mmap at an 8K boundary.
     * Check to see if the *end* of the mapping would collide with
     * an already-allocated page.  If so, we chop the mmap off at
     * the 8K boundary and read in the page.  Otherwise, we just
     * mmap the whole thing...
     */
    if(end4k != end8k) {
	unsigned long	trailing_frag_start, trailing_frag_len;
	unsigned long	trailing_frag_offset;

	pagebits = get_page_bits(end4k);
	protbits = pagebits & PROT_BITS;
	if(pagebits & PAGE_VALID) {
	    trailing_frag_start = ALPHA_PAGE_ROUND_DOWN(end);
	    trailing_frag_len = end - trailing_frag_start + 1;
	    trailing_frag_offset = offset + length - trailing_frag_len;

	    /* Yes, we collide with an existing allocation.  If
	     * necessary, read in the data.  Adjust our permissions
	     * and boundaries accordingly.
	     */
	    if(!(prot & MAP_ANONYMOUS)) {
		if(!(protbits & PROT_WRITE)) {
		    sys_mprotect(trailing_frag_start, 
				 ALPHA_PAGE_SIZE, 
				 protbits | PROT_WRITE);
		}
		if(debug_mmap) {
		   fprintf(x86_logfile, "MMAP: Reading in trailing fragment\n");
		   fprintf(x86_logfile, "offset 0x%x addr 0x%x, length 0x%x\n",
			trailing_frag_offset, trailing_frag_start,
			trailing_frag_len);
		   fflush(x86_logfile);
		}
		lseek(fd, trailing_frag_offset, SEEK_SET);
		read(fd, trailing_frag_start, trailing_frag_len);
	        if((protbits | PROT_WRITE) != (protbits | prot)) {
	    	    sys_mprotect(trailing_frag_start, 
				 ALPHA_PAGE_SIZE,
				 protbits | prot);
	        }
	    }
	    else {
		sys_mprotect(trailing_frag_start, ALPHA_PAGE_SIZE,
				protbits | prot);
	    }

	    /* In any case, we can now rein in the core allocation */
	    if(length <= trailing_frag_len) {
		/* Again, we're done! */
		if(debug_mmap) {
		    fprintf(x86_logfile, "-------------------------------------\n");
		    fflush(x86_logfile);
		}
		return(start);
	    }
	    else {
	        length -= trailing_frag_len;
	    }
	}
	if(debug_mmap) {
	    fprintf(x86_logfile, "After dealing w/trailing frag:\n");
	    fprintf(x86_logfile, "  offset=0x%x, length=0x%x, start8k=0x%x\n",
			offset, length, start8k);
	}
    }

    /* Now we can mmap the "middle" of this piece... */
    mmap_result = sys_mmap(start8k, length, prot, flags, fd, offset);
    if((long)mmap_result == -1) {
	perror("core mmap");
	if(debug_mmap) {
	    fprintf(x86_logfile, "-------------------------------------\n");
	    fflush(x86_logfile);
	}
	return(-1);
    }

    /* Add this mapping */
    set_page_range_bits(start8k, ALPHA_PAGE_ROUND_UP(start8k+length)-1, 
				prot | PAGE_VALID);
    /* done! */
    if(debug_mmap) {
	fprintf(x86_logfile, "-------------------------------------\n");
	fflush(x86_logfile);
    }
    return(start);
}

/* Grow the stack to include the specified address.  Actually, we'll give
 * a generous allocation beyond that so we don't have to keep coming back
 * for more...
 */
void grow_stack(unsigned long newsp)
{
    unsigned long   new_stktop;
    int retval;

    new_stktop = ALPHA_PAGE_ROUND_DOWN(newsp);
    new_stktop -= (8 * ALPHA_PAGE_SIZE);

    retval = mmap4k(new_stktop, stktop - new_stktop, PROT_READ|PROT_WRITE,
			MAP_FIXED | MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);

    if(retval < 0) {
	perror("Cannot grow stack!\n");
	exit(-1);
    }
    else {
	if(debug_mmap) {
	    fprintf(x86_logfile, "Grew stack to 0x%x\n", new_stktop);
	    fflush(x86_logfile);
	}
    }
}

