/*
 * ioctl.c
 *
 * by Adam Stachelek
 *
 * Table-driven ioctl processing for em86 Linux Intel->Alpha emulator
 */

#include <ioctl_thunk.h>
#include <asm/termiobits.h>
#define __KERNEL__
#include <linux/kdev_t.h>
#undef __KERNEL__
#include <linux/fs.h>
#define __KERNEL__
#include <linux/serial.h>
#include <linux/tty.h>
#undef __KERNEL__
#include <linux/route.h>
#include <linux/socket.h>
#include <linux/if_arp.h>
#include <errno.h>
#include <termios.h>
#include <stdio.h>
#include <stdlib.h>
#include <ioctls.h>

int alpha_to_intel_errno (int e);
unsigned int x86_to_alpha_bitmask(unsigned int, bitmask_transtbl *);
unsigned int alpha_to_x86_bitmask(unsigned int, bitmask_transtbl *);

inline void intel_to_alpha_termios (struct termios *x86, struct termios *alpha);
inline void alpha_to_intel_termios (struct termios *alpha, struct termios *x86);
void conv_struct (char *x86, char *alpha, argtype list [], char toalpha);

#define intel_to_alpha_struct(x86, alpha, list) conv_struct (x86, alpha, list, 1)
#define alpha_to_intel_struct(x86, alpha, list) conv_struct (alpha, x86, list, 0)

#define GET_TYPE_SIZE(t) (\
(t == TYPE_CHAR)           ? sizeof (char) :\
(t == TYPE_SHORT)          ? sizeof (short) :\
(t == TYPE_INT)            ? sizeof (int) :\
(t == TYPE_LONG)           ? sizeof (long) :\
(t == TYPE_ULONG)          ? sizeof (unsigned long) :\
(t == TYPE_PTR)            ? sizeof (unsigned long) :\
(t == TYPE_TERMIOS)        ? sizeof (struct termios) :\
(t == TYPE_WINSIZE)        ? sizeof (struct winsize) :\
(t == TYPE_TQ_STRUCT)      ? sizeof (struct tq_struct) :\
(t == TYPE_ASYNC_STRUCT)   ? sizeof (struct async_struct) :\
(t == TYPE_ASYNC_ICOUNT)   ? sizeof (struct async_icount) :\
(t == TYPE_TTY_STRUCT)     ? sizeof (struct tty_struct) :\
(t == TYPE_TTY_DRIVER)     ? sizeof (struct tty_driver) :\
(t == TYPE_TTY_LDISC)      ? sizeof (struct tty_ldisc) :\
(t == TYPE_TTY_FLIP_BUFFER)? sizeof (struct tty_flip_buffer) :\
(t == TYPE_SERIAL_MULTIPORT_STRUCT)? sizeof (struct serial_multiport_struct) :\
(t == TYPE_SERIAL_ICOUNTER_STRUCT)? sizeof (struct serial_icounter_struct) :\
(t == TYPE_SOCKADDR)       ? sizeof (struct sockaddr) :\
(t == TYPE_RTENTRY)        ? sizeof (struct rtentry) :\
(t == TYPE_IFMAP)          ? sizeof (struct ifmap) :\
(t == TYPE_SOCKADDR_IFREQ) ? sizeof (struct ifreq) :\
(t == TYPE_SHORT_IFREQ)    ? sizeof (struct ifreq) :\
(t == TYPE_INT_IFREQ)      ? sizeof (struct ifreq) :\
(t == TYPE_IFMAP_IFREQ)    ? sizeof (struct ifreq) :\
(t == TYPE_CHAR_IFREQ)     ? sizeof (struct ifreq) :\
(t == TYPE_PTR_IFREQ)      ? sizeof (struct ifreq) :\
(t == TYPE_IFCONF)         ? sizeof (struct ifconf) :\
(t == TYPE_ARPREQ)         ? sizeof (struct arpreq) :\
(t == TYPE_ARPREQ_OLD)     ? sizeof (struct arpreq_old) :\
TYPE_NULL)

#define GET_TYPE_ASSIGN(t, a, b) (\
(t == TYPE_CHAR)           ? *(char *) a = *(char *) b :\
(t == TYPE_SHORT)          ? *(short *) a = *(short *) b :\
(t == TYPE_INT)            ? *(int *) a = *(int *) b :\
(t == TYPE_LONG)           ? *(long *) a = *(long *) b :\
(t == TYPE_ULONG)          ? *(unsigned long *) a = *(unsigned long *) b :\
(t == TYPE_PTR)            ? *(unsigned long *) a = *(unsigned long *) b :\
TYPE_NULL)

extern FILE * x86_logfile;

int do_ioctl (int a0, int a1, int a2, int a3, int a4, int a5)
{
  int fd = a0;
  int x86_cmd = a1;
  ioctl_descriptor *id;

  for (id = ioctl_table; id->x86_cmd; id++)
    if (x86_cmd == id->x86_cmd)
      break;

  if (id->x86_cmd == 0) {
    fprintf (x86_logfile, "Unrecognized ioctl code 0x%x\n", x86_cmd);
    return (-EINVAL);
  }

  return (id->routine) (fd, *id, a2);
}

int do_struct_ioctl (int fd, ioctl_descriptor desc, int arg)
{
  int retval;
  char *alpha;
  alpha = (char *) malloc (MAX_STRUCT_SIZE);
  
  switch (desc.arguse)
    {
    case _IOC_WRITE:
      intel_to_alpha_struct 
	((char *) intel_to_alpha_ptr (arg), alpha, desc.list);
      retval = ioctl (fd, desc.alpha_cmd, alpha);
      if (retval < 0)
	return (-alpha_to_intel_errno (errno));
      free (alpha);
      return (retval);
    case _IOC_READ:
      retval = ioctl (fd, desc.alpha_cmd, alpha);
      if (retval < 0)
	return (-alpha_to_intel_errno (errno));
      alpha_to_intel_struct 
	(alpha, (char *) intel_to_alpha_ptr (arg), desc.list);
      free (alpha);
      return (retval);
    case _IOC_WRITE | _IOC_READ:
      intel_to_alpha_struct 
	((char *) intel_to_alpha_ptr (arg), alpha, desc.list);
      retval = ioctl (fd, desc.alpha_cmd, alpha);
      if (retval < 0)
	return (-alpha_to_intel_errno (errno));
      alpha_to_intel_struct 
	(alpha, (char *) intel_to_alpha_ptr (arg), desc.list);
      free (alpha);
      return (retval);
    }
}

int do_termios_ioctl (int fd, ioctl_descriptor desc, int arg)
{
  struct termios alpha_termios;
  struct termios *x86_termios;
  int retval;

  x86_termios = (struct termios *) intel_to_alpha_ptr (arg);
  switch (desc.arguse)
    {
    case _IOC_WRITE:
      intel_to_alpha_termios (x86_termios, &alpha_termios);
      retval = ioctl (fd, desc.alpha_cmd, &alpha_termios);
      if (retval < 0)
	return (-alpha_to_intel_errno (errno));
      return (retval);
    case _IOC_READ:
      retval = ioctl (fd, desc.alpha_cmd, &alpha_termios);
      if (retval < 0)
	return (-alpha_to_intel_errno (errno));
      alpha_to_intel_termios (&alpha_termios, x86_termios);
      return (retval);
    case (_IOC_WRITE | _IOC_READ):
      intel_to_alpha_termios (x86_termios, &alpha_termios);
      retval = ioctl (fd, desc.alpha_cmd, &alpha_termios);
      if (retval < 0)
	return (-alpha_to_intel_errno (errno));
      alpha_to_intel_termios (&alpha_termios, x86_termios);
      return (retval);
    }
}

int do_ptr_ioctl (int fd, ioctl_descriptor desc, int arg)
{
  int retval = ioctl (fd, desc.alpha_cmd, (int *) intel_to_alpha_ptr (arg));
  if (retval < 0)
    return (-alpha_to_intel_errno (errno));
  return (retval);
}

int do_int_ioctl (int fd, ioctl_descriptor desc, int arg)
{
  int retval = ioctl (fd, desc.alpha_cmd, arg);
  if (retval < 0)
    return (-alpha_to_intel_errno (errno));
  return (retval);
}

int do_longptr_ioctl (int fd, ioctl_descriptor desc, int arg)
{
  int retval;
  unsigned long *alpha;
  
  alpha = (unsigned long *) intel_to_alpha_ptr (arg);
  *alpha = intel_to_alpha_long (*alpha); 
  retval = ioctl (fd, desc.alpha_cmd, (int *) alpha);
  if (retval < 0)
    return (-alpha_to_intel_errno (errno));
  return (retval);
}

int do_void_ioctl (int fd, ioctl_descriptor desc, int arg)
{
  int retval = ioctl (fd, desc.alpha_cmd);
  if (retval < 0)
    return (-alpha_to_intel_errno (errno));
  return (retval);
}
  
void conv_struct (char *x86, char *alpha, argtype list [], char toalpha)
{
  unsigned long x86_addr, alpha_addr;
  int j, i, size, array_size, array_type;
  argtype tmplist [] = { TYPE_NULL, TYPE_NULL };

  x86_addr = (unsigned long) x86;
  alpha_addr = (unsigned long) alpha;

  for (i = 0; list [i] != TYPE_NULL; i++) {    
    switch (list [i])
      {
      case TYPE_LONG:
      case TYPE_ULONG:
      case TYPE_PTR:
	size = GET_TYPE_SIZE (list [i]);
	x86_addr = (x86_addr + sizeof (int) - 1) & ~(sizeof (int) - 1);
	alpha_addr = (alpha_addr + sizeof (long) - 1) & ~(sizeof (long) - 1);
	if (toalpha)
	  *(long *) alpha_addr = ((long) (*(int *) x86_addr)) & 0xffffffff;
	else
	  *(int *) x86_addr = *(long *) alpha_addr;
	x86_addr += sizeof (int);
	alpha_addr += sizeof (long);
	break;
      case TYPE_TERMIOS:
	size = GET_TYPE_SIZE (list [i]);
	if (toalpha)
	  intel_to_alpha_termios 
	    ((struct termios *) x86_addr, (struct termios *) alpha_addr);
	else
	  alpha_to_intel_termios
	    ((struct termios *) alpha_addr, (struct termios *) x86_addr);
	x86_addr += X86_TERMIOS_SIZE;
	alpha_addr += size;
	break;
      case TYPE_ARRAY:
	i++;
	array_type = GET_ARRAY_TYPE (list [i]);
	array_size = GET_ARRAY_SIZE (list [i]);
	size = GET_TYPE_SIZE (array_type);
	tmplist [0] = array_type;
	for (j = 0; j < array_size; j++) {
	  if (toalpha)
	    intel_to_alpha_struct 
	      ((char *) x86_addr, (char *) alpha_addr, tmplist);
	  else
	    alpha_to_intel_struct 
	      ((char *) alpha_addr, (char *) x86_addr, tmplist);
	  if (array_type == TYPE_TERMIOS)
	    x86_addr += X86_TERMIOS_SIZE;
	  else
	    x86_addr += size;
	  alpha_addr += size;
	}
	break;
      default:
	size = GET_TYPE_SIZE (list [i]);
	if (list [i] > REGSTRUCT) {
	  if (toalpha)
	    intel_to_alpha_struct 
	      ((char *) x86_addr, (char *) alpha_addr, struct_table [list [i] - REGSTRUCT - 1]);
	  else
	    alpha_to_intel_struct 
	      ((char *) alpha_addr, (char *) x86_addr, struct_table [list [i] - REGSTRUCT - 1]);
	  x86_addr += size;
	  alpha_addr += size;
	} else {
	  x86_addr = (x86_addr + size - 1) & ~(size - 1);
	  alpha_addr = (alpha_addr + size - 1) & ~(size - 1);
	  if (toalpha)
	    GET_TYPE_ASSIGN (list [i], alpha_addr, x86_addr);
	  else
	    GET_TYPE_ASSIGN (list [i], x86_addr, alpha_addr);
	  x86_addr += size;
	  alpha_addr += size;
	}
      }
  }
}


inline void intel_to_alpha_termios (struct termios *x86, struct termios *alpha)
{
  alpha->c_iflag = 
    x86_to_alpha_bitmask(x86->c_iflag, iflag_tbl);
  alpha->c_oflag = 
    x86_to_alpha_bitmask(x86->c_oflag, oflag_tbl);
  alpha->c_cflag = 
    x86_to_alpha_bitmask(x86->c_cflag, cflag_tbl);
  alpha->c_lflag = 
    x86_to_alpha_bitmask(x86->c_lflag, lflag_tbl);
  alpha->c_line = x86->c_line;
  
  alpha->c_cc[VINTR] = x86->c_cc[X86_VINTR]; 
  alpha->c_cc[VQUIT] = x86->c_cc[X86_VQUIT]; 
  alpha->c_cc[VERASE] = x86->c_cc[X86_VERASE];       
  alpha->c_cc[VKILL] = x86->c_cc[X86_VKILL]; 
  alpha->c_cc[VEOF] = x86->c_cc[X86_VEOF];   
  alpha->c_cc[VTIME] = x86->c_cc[X86_VTIME]; 
  alpha->c_cc[VMIN] = x86->c_cc[X86_VMIN];   
  alpha->c_cc[VSWTC] = x86->c_cc[X86_VSWTC]; 
  alpha->c_cc[VSTART] = x86->c_cc[X86_VSTART];       
  alpha->c_cc[VSTOP] = x86->c_cc[X86_VSTOP]; 
  alpha->c_cc[VSUSP] = x86->c_cc[X86_VSUSP]; 
  alpha->c_cc[VEOL] = x86->c_cc[X86_VEOL];   
  alpha->c_cc[VREPRINT] = x86->c_cc[X86_VREPRINT];   
  alpha->c_cc[VDISCARD] = x86->c_cc[X86_VDISCARD];   
  alpha->c_cc[VWERASE] = x86->c_cc[X86_VWERASE];     
  alpha->c_cc[VLNEXT] = x86->c_cc[X86_VLNEXT];       
  alpha->c_cc[VEOL2] = x86->c_cc[X86_VEOL2]; 
}
  
inline void alpha_to_intel_termios (struct termios *alpha, struct termios *x86)
{
  x86->c_iflag = 
    alpha_to_x86_bitmask(alpha->c_iflag, iflag_tbl);
  x86->c_oflag = 
    alpha_to_x86_bitmask(alpha->c_oflag, oflag_tbl);
  x86->c_cflag = 
    alpha_to_x86_bitmask(alpha->c_cflag, cflag_tbl);
  x86->c_lflag = 
    alpha_to_x86_bitmask(alpha->c_lflag, lflag_tbl);
  x86->c_line = alpha->c_line;
  
  x86->c_cc[X86_VINTR] = alpha->c_cc[VINTR];
  x86->c_cc[X86_VQUIT] = alpha->c_cc[VQUIT];
  x86->c_cc[X86_VERASE] = alpha->c_cc[VERASE];
  x86->c_cc[X86_VKILL] = alpha->c_cc[VKILL];
  x86->c_cc[X86_VEOF] = alpha->c_cc[VEOF];
  x86->c_cc[X86_VTIME] = alpha->c_cc[VTIME];
  x86->c_cc[X86_VMIN] = alpha->c_cc[VMIN];
  x86->c_cc[X86_VSWTC] = alpha->c_cc[VSWTC];
  x86->c_cc[X86_VSTART] = alpha->c_cc[VSTART];
  x86->c_cc[X86_VSTOP] = alpha->c_cc[VSTOP];
  x86->c_cc[X86_VSUSP] = alpha->c_cc[VSUSP];
  x86->c_cc[X86_VEOL] = alpha->c_cc[VEOL];
  x86->c_cc[X86_VREPRINT] = alpha->c_cc[VREPRINT];
  x86->c_cc[X86_VDISCARD] = alpha->c_cc[VDISCARD];
  x86->c_cc[X86_VWERASE] = alpha->c_cc[VWERASE];
  x86->c_cc[X86_VLNEXT] = alpha->c_cc[VLNEXT];
  x86->c_cc[X86_VEOL2] = alpha->c_cc[VEOL2];
}

  





